//
//  GlyphsFileFormatProtocol.h
//  GlyphsCore
//
//  Created by Georg Seifert on 08.05.09.
//  Copyright 2009 schriftgestaltung.de. All rights reserved.
//

@class GSFont;
@protocol GlyphsFileFormat;

@protocol GlyphsProgressDelegate

- (NSWindow *)windowForProgressSheet;

- (void)cancelExport;

@end

@protocol GlyphsProgressProtocol

@property (unsafe_unretained, nonatomic) NSObject<GlyphsProgressDelegate> *handler;

- (void)showProgress;

- (void)hideProgress;

@end

/** GlyphsFileFormat: The protocol defining the file format plugin interface

 */
@protocol GlyphsFileFormat
/// The interface version.
@property (readonly) NSUInteger interfaceVersion;
/// The title of the plugin.
@property (readonly) NSString *title;
/// The view to be displayed in the export dialog.
@property (readonly, nonatomic) NSView *exportSettingsView;

/** The GSFont object is assigned the the plugin prior to the export.

 This is used to publish the export dialog.
 */
@property (strong, nonatomic) GSFont *font;

#ifndef GLYPHS_VIEWER

/** Exports a Font object.
 This function should ask the user for the place to save the store the font.

 Eventually errors have to be presented by the plugin. Use `[font.parent presentError:error];`
 @param font The font to export.

 */
- (void)exportFont:(GSFont *)font;

/** Saves a Font object to the specified URL.

 This is a save operation so it should be able to read the font back in. Otherwise use exportFont:

 @param font The font to export.
 @param URL The URL to save the font to.
 @param error On return, If the document contents could not be read, a pointer to an error object that encapsulates the reason they could not be read.
 @return YES if the operation was successful; otherwise, NO.
 */
- (BOOL)writeFont:(GSFont *)font toURL:(NSURL *)URL error:(out NSError **)error;

#endif

/** Reads a Font object from the specified URL.

 @param URL The URL to read the font from.
 @param type The UTI of the document
 @param error On return, If the document contents could not be read, a pointer to an error object that encapsulates the reason they could not be read.
 @return The font object, or nil if an error occurred.
 */
- (GSFont *)fontFromURL:(NSURL *)URL ofType:(NSString *)type error:(out NSError **)error;

/// The group ID defines the position of the exporter in the toolbar of the export window.
@property (readonly) NSUInteger groupID;

/// The name in the toolbar of the export window.
@property (readonly) NSString *toolbarTitle;

/// The name of the icon for the tab bar.
@property (readonly) NSString *toolbarIconName;

@optional

/// do main initialization
- (void)loadPlugin;

@property (nonatomic, assign) NSDictionary *exportOptions;

/** Import some Data into the existing Font object

 @param font  an existing, already open Font
 @param URL   the file to import from
 @param error on return

 @return True if success, False if failure
 */
- (BOOL)importFile:(GSFont *)font fromURL:(NSURL *)URL error:(out NSError **)error;

/// is called by the app when there could be a progress window.
- (void)showProgress;

/// Implement this if you support cancel
- (void)cancelExport;

@property (strong) NSObject<GlyphsProgressProtocol> *progressController;

@end
